<?php

namespace ElephantIO;

/**
 * ElephantIOClient is a rough implementation of socket.io protocol.
 * It should ease you dealing with a socket.io server.
 *
 * @author Ludovic Barreca <ludovic@balloonup.com>
 */
class Client {
    const TYPE_DISCONNECT   = 0;
    const TYPE_CONNECT      = 1;
    const TYPE_HEARTBEAT    = 2;
    const TYPE_MESSAGE      = 3;
    const TYPE_JSON_MESSAGE = 4;
    const TYPE_EVENT        = 5;
    const TYPE_ACK          = 6;
    const TYPE_ERROR        = 7;
    const TYPE_NOOP         = 8;

    private $socketIOUrl;
    private $serverHost;
    private $serverPort = 80;
    private $session;
    private $fd;
    private $buffer;
    private $lastId = 0;
    private $read;

    private $debug;

    public function __construct($socketIOUrl, $socketIOPath = 'socket.io', $protocol = 1, $read = true, $debug = false) {
        $this->socketIOUrl = $socketIOUrl.'/'.$socketIOPath.'/'.(string)$protocol;
        $this->read = $read;
        $this->debug = $debug;
        $this->parseUrl();
    }

    /**
     * Initialize a new connection
     *
     * @param boolean $keepalive
     * @return ElephantIOClient
     */
    public function init($keepalive = false) {
        $this->handshake();
        $this->connect();
        if ($keepalive) {
            $this->keepAlive();
        } else {
            return $this;
        }
    }

    /**
     * Keep the connection alive and dispatch events
     *
     * @access public
     * @todo work on callbacks
     */
    public function keepAlive() {
        while(true) {
            if ($this->session['heartbeat_timeout'] > 0 && $this->session['heartbeat_timeout']+$this->heartbeatStamp-5 < time()) {
                $this->send(self::TYPE_HEARTBEAT);
                $this->heartbeatStamp = time();
            }

            $r = array($this->fd);
            $w = $e = null;

            if (stream_select($r, $w, $e, 5) == 0) continue;

            $this->read();
        }
    }

    /**
     * Read the buffer and return the oldest event in stack
     *
     * @access public
     * @return string
     */
    public function read() {
        while(true) {
            if (strlen($this->buffer) > 1) {
                if ($this->buffer[0] != "\x00") {
                    $this->buffer = (string)substr($this->buffer, 1);
                    continue;
                }

                $pos = strpos($this->buffer, "\xff");
            } else {
                $pos = false;
            }

            if ($pos === false) {
                $tmp = fread($this->fd, 4096);

                if ($tmp === false) {
                    throw new \Exception('Something went wrong. Socket seems to be closed !');
                }

                $this->buffer .= $tmp;
                continue;
            }
            $res = substr($this->buffer, 1, $pos-1);
            $this->buffer = (string)substr($this->buffer, $pos+1);
            $this->stdout('debug', 'Received '.$res);

            return $res;
        }
    }

    /**
     * Send message to the websocket
     *
     * @access public
     * @param int $type
     * @param int $id
     * @param int $endpoint
     * @param string $message
     * @return ElephantIO\Client
     */
    public function send($type, $id = null, $endpoint = null, $message = null) {
        if (!is_int($type) || $type > 8) {
            throw new \InvalidArgumentException('ElephantIOClient::send() type parameter must be an integer strictly inferior to 9.');
        }

        fwrite($this->fd, "\x00".$type.":".$id.":".$endpoint.":".$message."\xff");
        $this->stdout('debug', 'Sent '.$type.":".$id.":".$endpoint.":".$message);
        usleep(300*1000);

        return $this;
    }


    /**
     * Emit an event
     *
     * @param string $event
     * @param array $args
     * @param string $endpoint
     * @param function $callback - ignored for the time being
     * @todo work on callbacks
     */
    public function emit($event, $args, $endpoint, $callback = null) {
        $this->send(5, null, $endpoint, json_encode(array(
            'name' => $event,
            'args' => $args,
            )
        ));
    }

    /**
     * Close the socket
     *
     * @return boolean
     */
    public function close()
    {
        if ($this->fd) {
            fclose($this->fd);

            return true;
        }

        return false;
    }

    /**
     * Send ANSI formatted message to stdout.
     * First parameter must be either debug, info, error or ok
     *
     * @access public
     * @param string $type
     * @param string $message
     */
    public function stdout($type, $message) {
        if (!defined('STDOUT') || !$this->debug) {
            return false;
        }

        $typeMap = array(
            'debug'   => array(36, '- debug -'),
            'info'    => array(37, '- info  -'),
            'error'   => array(31, '- error -'),
            'ok'      => array(32, '- ok    -'),
        );

        if (!array_key_exists($type, $typeMap)) {
            throw new \InvalidArgumentException('ElephantIOClient::stdout $type parameter must be debug, info, error or success. Got '.$type);
        }

        fwrite(STDOUT, "\033[".$typeMap[$type][0]."m".$typeMap[$type][1]."\033[37m  ".$message."\r\n");
    }

    /**
     * Handshake with socket.io server
     *
     * @access private
     * @return bool
     */
    private function handshake() {
        $ch = curl_init($this->socketIOUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $res = curl_exec($ch);

        if ($res === false) {
            throw new \Exception(curl_error($ch));
        }

        $sess = explode(':', $res);
        $this->session['sid'] = $sess[0];
        $this->session['heartbeat_timeout'] = $sess[1];
        $this->session['connection_timeout'] = $sess[2];
        $this->session['supported_transports'] = array_flip(explode(',', $sess[3]));
        if (!isset($this->session['supported_transports']['websocket'])) {
            throw new \Exception('This socket.io server do not support websocket protocol. Terminating connection...');
        }

        return true;
    }

    /**
     * Connects using websocket protocol
     *
     * @access private
     * @return bool
     */
    private function connect() {
        $this->fd = fsockopen($this->serverHost, $this->serverPort, $errno, $errstr);

        if (!$this->fd) {
            throw new \Exception('fsockopen returned: '.$errstr);
        }

        $out  = "GET /socket.io/1/websocket/".$this->session['sid']." HTTP/1.1\r\n";
        $out .= "Upgrade: WebSocket\r\n";
        $out .= "Connection: Upgrade\r\n";
        $out .= "Host: ".$this->serverHost."\r\n";
        $out .= "Origin: *\r\n\r\n";
        fwrite($this->fd, $out);
        $res = fgets($this->fd);

        if ($res === false) {
            throw new \Exception('Socket.io did not respond properly. Aborting...');
        }

        if ($subres = substr($res, 0, 12) != 'HTTP/1.1 101') {
            throw new \Exception('Unexpected Response. Expected HTTP/1.1 101 got '.$subres.'. Aborting...');
        }

        while(true) {
            $res = trim(fgets($this->fd));
            if ($res === '') break;
        }

        if ($this->read) {
            if ($this->read() != '1::') {
                throw new \Exception('Socket.io did not send connect response. Aborting...');
            } else {
                $this->stdout('info', 'Server report us as connected !');
            }
        }

        $this->send(self::TYPE_CONNECT);
        $this->heartbeatStamp = time();
    }

    /**
     * Parse the url and set server parameters
     *
     * @access private
     * @return bool
     */
    private function parseUrl() {
        $url = parse_url($this->socketIOUrl);
        $this->serverHost = $url['host'];
        $this->serverPort = isset($url['port']) ? $url['port'] : null;

        if ($url['scheme'] == 'https') {
            $this->serverHost = 'ssl://'.$this->serverHost;
            if (!$this->serverPort) {
                $this->serverPort = 443;
            }
        }

        return true;
    }
}
